/*
    $Id: cled.c,v 2.5 90/08/26 11:32:08 sw Exp $

    A simple line editor intended for command line editing on Xenix
    with a TCAP type terminal using the line discipline mode of
    terminal operation.

    Authors:

    Piercarlo Grandi (piercarl). Aberystwyth. (pcg@cs.aber.ac.uk):

	substantially cleaned up the code and made it more modular
	and readable. More configurable, with an EMACS key bindings
	options for non DEC VT compatible terminals. Able to work
	on dumb terminals if necessary. Corrected a few bugs.

    Dave Shepperd (dms). Atari Games, corp. (shepperd@dms.UUCP):

	hacked lvr's editor to bits to make it re-entrant and added
	the line discipline and device interfaces for SCO Xenix/386.

    Lyle Rains (lvr). Atari Games, corp. (rains@dms.UUCP):

	wrote the editor in 1983 or so for a RT-11 system intended to be
	included as a library module.

    Copyright 1989,1990 Piercarlo Grandi (parts).
    Copyright 1989 Atari Games, Corp. (most).

    Reproduction, adaptation, distribution, performance or display of
    this computer program or the associated documentation is not
    prohibited provided this copyright notice accompanies said
    reproduction, adaptation, distribution, performance or display.
*/

#include <sys/types.h>
#include <sys/tty.h>
#include <sys/termio.h>

#include "patchlevel.h"
#include "sys/cledio.h"
#include "sys/cled.h"

#include <sys/param.h>
#include <sys/sysmacros.h>
#include <sys/conf.h>
#include <sys/errno.h>

#ifdef M_KERNEL
#   undef INKERNEL
#   define INKERNEL	1
#endif

#ifdef INKERNEL
#   undef M_KERNEL
#   define M_KERNEL	1
#endif

#if INKERNEL
#   if M_UNIX
#	include <sys/immu.h>
#	include <sys/region.h>
#   else
#	include <sys/mmu.h>
#   endif
#   if M_I386
#	include <sys/page.h>
#	include <sys/seg.h>
#   endif
#   include <sys/proc.h>
#   include <sys/var.h>
#   include <sys/dir.h>
#   include <sys/signal.h>
#   include <sys/user.h>
#   if M_UNIX && defined(ftop)
#	undef ftop
#   endif
#   include <sys/ioctl.h>
#   include <sys/file.h>
#   if M_UNIX
#	include <sys/cmn_err.h>
#   endif
#endif

extern int		cle_ttys;
extern int		cle_leds;

extern struct led_buf	cle_buffers[];
extern struct tty_buf	cle_ttybuf[];

/*
    Control characters
*/

#define BEL		007
#define ESC		033
#define RUB		0177

/*
    Parser states
*/

#define NORMAL		0x0000
#define ESCAPE		0x0001
#define ANSIKEY		0x0002
#define QUOTE		0x0003
#define NCC_SPC(c)	(0x0100+(c))

/*
    Boolean type and values
*/

#define FLAG		unchar
 
#define YES		1
#define NO		0
#define DEL		1
#define NODEL		0

#if INKERNEL
#   define cle_putc(c,tp)	putc((c),&(tp)->t_outq)
#   define cle_getc(tp)		getc(&(tp)->t_rawq)
#else
#   include <stdio.h>
#   include <malloc.h>
#   define cle_puts(str,tp)	if (str) fputs((str),stderr)
#   define cle_putc(chr,tp)	fputc((chr),stderr)
#   define cle_getc(tp)		fgetc(stdin)
#endif

#define MIN(a,b)	((a) < (b) ? (a) : (b))
#define history_end(lb)	((lb)->historyend-1)

static unchar		our_lineno;

/*
    This routine is a dummy routine that simply marks the start of cled in
    the map file (for trouble shooting)
*/

void			cled_begin(){}

/*
    The following n routines comprise the editor proper. The process
    context in all cases is task time. None of these routines are (nor
    should they ever be) called at interrupt time.
*/

static void		pfront();
static void		pback();
static void		bol();
static void		eol();

static void		reprint();
static void		ring_bell();
static void		kick_out();

static unchar		echo();

/*
    Put a string of text into the command buffer, echoing them as they are
    inserted.

    ENTRY: cp - ptr to string of chars to be copied cnt - count of chars to
    copy lb - ptr to led_buf into which to place the text

    EXIT: chars moved into lb->line[lb->lcurs]. lb->lcurs adjusted as
    required lb->rcurs adjusted if overstrike mode lb->flags may be set
    chars echoed as required
*/

static void             putt(cp,cnt,lb)
    register unchar         *cp;
    int                     cnt;
    struct led_buf          *lb;
{
    register unchar         **rc,**lc;
    int			    gap;
    int			    last;
    unsigned		    iflg;
    struct tty              *typ;

    rc = &lb->rcurs;
    lc = &lb->lcurs;

    typ = lb->tbp->ttyp;
    iflg = typ->t_iflag;

    /*
	We want the line not to grow beyond cols, including the prompt and
	an empty position at the end of the line to avoid problems with
	automargin terminals.
    */

    last = lb->tbp->cols - lb->promptlen - 1;
    gap = (MAXLINE-lb->tbp->cols + 1) + lb->promptlen + 1;

    while (cnt--)
    {
	register unchar		    ch = *cp;

	if (iflg&ISTRIP) ch &= 0x7F;

	if (ch != '\0')
	{
	    if ((iflg&IUCLC) && ch >= 'A' && ch <= 'Z')
		ch += 'a' - 'A';

	    if ((*rc - *lc) <= gap || (ch = echo(ch,lb,last)) == '\0')
		break;

	    *(*lc)++ = *cp++ = ch;
	    if (!(lb->flags&LD_INSERT) && *(*rc) != '\0')
		(*rc)++;

	    lb->flags |= LD_DIRTY;
	}
    }

    *(*lc) = '\0';

    if (cnt >= 0)
	ring_bell(lb);

    pback(lb,NO);
}

/*
    Find pointer to n'th string in a history or key define buffer.

    ENTRY: start - ptr into buffer where search is to begin ndx - the
    number of the string to find

    EXIT: returns ptr to first char of requested string if present else it
    returns ptr to null string at end of buffer.
*/

static unchar           *ptr(start,ndx)
    register unchar	    *start;
    register int	    ndx;
{
    while (ndx-- && *start)
	start += *start;

    return start;
}

/*
    Find the pointer to matching string in a history or key define buffer.

    ENTRY: src - ptr to place in buffer to begin the search.  patt - ptr
    to string to match against len - len of string pointed to by patt indx
    - ptr to int into which the number of the found string in the buffer
    is placed.  exact - if YES, the lengths must match as well as the
    pattern. if NO, only 'len' chars are compared.

    EXIT: returns ptr to first char in found string or ptr to null string
    at end of buffer if no match. *indx is updated if the string is found
    otherwise it is not changed.
*/

static unchar           *match_ptr(src,patt,len,indx,exact)
    unchar                  *src,*patt;
    int                     exact,len,*indx;
{
    int                     nxtndx;

    for
    (
	nxtndx = *indx, src = ptr(src,nxtndx);
	*src != '\0';
	nxtndx++, src += *src
    )
	if ((*src - 1) == len || (exact == NO && (*src - 1) > len))
	{
	    int                     result;
	    register int	    tlen;
	    register unchar	    *a,*b;

	    result = 0;

	    for (a = src+1, b = patt, tlen = len; tlen != 0; --tlen)
		if ((result = *a++ - *b++) != 0)
		    break;

	    if (result == 0)
		break;
	}

    if (*src == '\0')
	    src = (unchar *) '\0';
    else    *indx = nxtndx;

    return src;
}

/*
    Place n'th string from history or key definition buffer into command buff

    ENTRY: src - ptr to place in history or key def buffer to begin search
    ndx - number of string to pick up lb - ptr to led_buf into which to
    place the found string

    EXIT: requested string (or null string) is moved to command buf
    replacing whatever might be there already. A number of members of the
    lb struct will have been changed.
*/

static void             getstr(src,ndx,lb)
    register unchar         *src;
    int                     ndx;
    struct led_buf          *lb;
{
    register unsigned       cnt;

    src = ptr(src,ndx);
    if (cnt = (unsigned) *src++)
	putt(src,--cnt,lb);
}

/*
    Move the cursor n places to the left.

    ENTRY: cnt - number of column positions to move left delete - if 1,
    delete the characters under the cursor as it is moved if 0, don't
    delete the chars under the cursor lb - ptr to led_buf which holds the
    cursor info

    EXIT: cursor is moved left appropriately unless it is already at the
    left margin in which case nothing will happen. Characters may be
    echoed or the right half of the line may be redrawn if deleting. A
    number of members of the lb struct will have been changed.
*/

static void             curs_l(cnt,delete,lb)
    int                     cnt;
    FLAG                    delete;
    struct led_buf          *lb;
{
    register unchar         *rc,*lc;
    register FLAG           atab;
    register struct tty	    *typ;

    atab = NO;

    rc = lb->rcurs;
    lc = lb->lcurs;
    typ = lb->tbp->ttyp;

    while ((lc > lb->line) && cnt--)
    {
	if ((*--rc = *--lc) == '\t')
	    atab |= YES;

	lb->current--;
	if (!atab)
	    cle_putc('\b',typ);
    }
    *(lb->lcurs = lc) = '\0';

    if (atab)
	pfront(lb,NO);

    if (delete)
    {
	lb->flags |= LD_DIRTY;
	pback(lb,YES);
    }
    else
    {
	lb->rcurs = rc;
	if (atab)
	    pback(lb,NO);
    }
}

/*
    Move the cursor n places to the right.

    ENTRY: cnt - number of column positions to move right delete - if 1,
    delete the characters under the cursor as it is moved if 0, don't
    delete the chars under the cursor lb - ptr to led_buf which holds the
    cursor info

    EXIT: cursor is moved right appropriately unless it is already at the
    right margin in which case nothing will happen. Characters may be
    echoed or the right half of the line may be redrawn if deleting. A
    number of members of the lb struct will have been changed.
*/

static void             curs_r(cnt,delete,lb)
    int                     cnt;
    FLAG                    delete;
    struct led_buf          *lb;
{
    register unchar         *rc,*lc;

    rc = lb->rcurs;
    lc = lb->lcurs;

    if (!delete)
    {
	while (*rc && cnt--) *lc++ = echo(*rc++,lb,MAXLINE);
	lb->rcurs = rc;
	*(lb->lcurs = lc) = '\0';
    }
    else
    {
	while (*rc && cnt--) rc++;
	lb->rcurs = rc;
	lb->flags |= LD_DIRTY;
	pback(lb,YES);
    }
}

/*
    Test a char for being alpha-numeric.

    ENTRY: chr - chr to test

    EXIT: returns YES if chr is alpha-numeric or '_'
*/

static int              isaln(chr)
    register unchar         chr;
{
    return (
	(chr >= 'a' && chr <= 'z')
	|| (chr >= 'A' && chr <= 'Z')
	|| (chr >= '0' && chr <= '9')
	|| chr == '_'
    );
}

/*
    Test a char for being white-space.

    ENTRY: chr - chr to test

    EXIT: returns YES if chr is white-space
*/

static int              isws(chr)
    register unchar         chr;

{
    return (chr == ' ' || chr == '\t' || chr == '\n');
}

/*
    Test a char for being delimiter.

    ENTRY: chr - chr to test

    EXIT: returns YES if chr is delimiter
*/

static int              isdelim(chr)
    register unchar         chr;
{
    return (!isaln(chr) && !isws(chr));
}

/*
    Skip to end of "word" on the left and optionally 'eat' it.

    ENTRY: lb - ptr to led_buf

    EXIT: cursor moved. "word" skipped
*/

static void             skipwl(lb)
    struct led_buf          *lb;
{
    register unchar         *cp;

    cp = lb->lcurs;

    while (cp > lb->line && isws(*(cp-1)))	--cp;
    while (cp > lb->line && isaln(*(cp-1)))	--cp;
    while (cp > lb->line && isdelim(*(cp-1)))	--cp;

    curs_l(lb->lcurs - cp,NO,lb);
}

/*
    Skip to end of "word" on the right and optionally 'eat' it.

    ENTRY: lb - ptr to led_buf

    EXIT: cursor moved. "word" either skipped or eaten
*/

static void             skipwr(lb)
    struct led_buf          *lb;
{
    register unchar         *cp;

    cp = lb->rcurs;

    while (*cp && isdelim(*cp))		++cp;
    while (*cp && isaln(*cp))		++cp;
    while (*cp && isws(*cp))		++cp;

    curs_r(cp - lb->rcurs,NO,lb);
}

/*
    Delete 'word' to left of cursor

    ENTRY: lb - ptr to led_buf containing cmd_buf

    EXIT: 'word' to left of cursor is deleted unless cursor is already at
    left margin in which case nothing happens. Several members of the lb
    struct are updated.
*/

static void		d_lwrd(lb)
    struct led_buf          *lb;
{
    register unchar         *cp;

    cp = lb->lcurs;

    while (cp > lb->line && isws(*(cp - 1)))	--cp;
    curs_l(lb->lcurs - cp,NO,lb);

    while (cp > lb->line && isaln(*(cp - 1)))	--cp;
    while (cp > lb->line && isdelim(*(cp - 1)))	--cp;
    curs_l(lb->lcurs - cp,YES,lb);
}

/*
    Delete 'word' to right of cursor

    ENTRY: lb - ptr to led_buf containing cmd_buf

    EXIT: 'word' to right of cursor is deleted unless cursor is already at
    right margin in which case nothing happens. Several members of the lb
    struct are updated.
*/

static void              d_rwrd(lb)
    struct led_buf          *lb;
{
    register unchar         *cp;

    cp = lb->rcurs;

    while (*cp && isws(*cp))	    ++cp;
    curs_r(cp - lb->rcurs,NO,lb);

    while (*cp && isdelim(*cp))	    ++cp;
    while (*cp && isaln(*cp))	    ++cp;
    curs_r(cp - lb->rcurs,YES,lb);
}

/*
    Copy current contents of command buf to key def buffer

    ENTRY: cp - ptr to place in key def buffer to deposit cmd buf len -
    number of chars to move lb - ptr to led_buf containing the cmd buf At
    exit: command string is moved to requested spot in buffer Strings in
    the history buffer may be deleted if there's overlap. A number of
    members of the lb struct will have been changed.
*/

static void             copycom(cp,len,lb)
    register unchar         *cp;
    register                len;
    struct led_buf          *lb;
{
    register unchar         *cp2;

    if (len > (TTYHOG-3))
	return;

    cp2 = lb->line;
    *cp++ = len;
    while (--len)
	*cp++ = *cp2++;

    /* eliminate oldlines if overwritten */
    for (cp = lb->history; *cp && (cp + *cp) <= history_end(lb); cp += *cp);

    *cp = '\0';
}

/*
    Complete the read (insert a newline in command buffer)

    ENTRY: lb - ptr to led_buf containing cmd_buf

    EXIT: lb struct is cleaned up and a LD_DONE is set in flags to
    indicate the read is complete.  The contents of the command line is
    inserted in the history buffer. If it matches a line already in the
    history buffer, the dup is removed and the new one is inserted at the
    front. Several members of the lb struct are updated.
*/

static void             newline(lb)
    struct led_buf          *lb;
{
    register unchar         *end;
    register unchar         *cp;
    int                     linelen;

    eol(lb);
    linelen = (lb->lcurs - lb->line) + 1;

    if (linelen > 1 && linelen < ((history_end(lb) - lb->history) - 1))
    {
	/*
	   If dirty = NO, then we used an old line without modification so
	   we'll squeeze it out of the old buffer to eliminate the
	   repetition, and recopy it into the front of the old buffer. This
	   allows us to hold older commands longer. If dirty = YES, then
	   we'll look for a matching command in the buffer and, if one is
	   found, move it to the head of the buffer. If not, then we just
	   insert the new string at the head of the buffer.
	*/

	if (!(lb->flags&LD_DIRTY))
	    end = ptr(lb->history,lb->lastline);
	else
	{
	    lb->lastline = 0;
	    end = match_ptr(lb->history,lb->line,linelen - 1,&lb->lastline,YES);
	    if (end == 0)
		end = lb->historyend - linelen;
	    lb->matchlen = linelen;
	}

	for (cp = end + linelen; end > (lb->history + 1); *--cp = *--end);
	copycom(lb->history + 1,linelen,lb);
    }

    lb->flags |= LD_DONE;
}

#if INKERNEL

/*
    Kick start the output. This routine is called whenever there is data in
    the t_outq buffer to make sure that it gets sent to the terminal.

    ENTRY: Process context: Task or Interrupt. May be called from either.
    tp - ptr to tty struct

    EXIT: If data present in t_outq and output not already running, then a
    call to the terminal driver processor routine (t_proc) with the output
    option selected is made which will cause data to be moved from the
    t_outq clist to the t_obuf output array for subsequent output to the
    hardware.
*/

static void             kick_out(tp)
    struct tty              *tp;
{
    int                     ospl;

    ospl = spl6();
    {
	if (tp->t_outq.c_cc != 0
	    && (!(tp->t_state & (BUSY|TIMEOUT|TTSTOP))
		|| tp->t_tbuf.c_count == 0))
	    {
		splx(ospl);
		(*tp->t_proc) (tp,T_OUTPUT);
	    }
    }
    splx(ospl);
}

/*
    Put a null terminated string of characters in the output que for
    subsequent output to the terminal.

    ENTRY: Process context: Task or Interrupt. s - ptr to null terminated
    string to output tp - ptr to tty struct

    EXIT: characters are placed in the t_outq clist. If there's no room
    for the whole message, that that won't fit is discarded with no error.
*/

static void             cle_puts(s,tp)
    unchar                  *s;
    struct tty              *tp;
{
    register unchar         *s1 = s;
    struct clist            dumcl,*clp;
    struct cblock           *srcp;
    int                     ospl,cnt;

    if (s1 == 0 || *s1 == 0)
	return;

    while (*s1++);
    cnt = s1-1 - s;

    dumcl.c_cc = 0;
    dumcl.c_cf = dumcl.c_cl = 0;
    clp = &tp->t_outq;

    ospl = spl6();
    {
	putcbp(&dumcl,s,cnt);
	while ((srcp = getcb(&dumcl)) != 0)
	    putcb(srcp,clp);
    }
    splx(ospl);
}
#endif

/*
    Echo a char to the terminal

    ENTRY: chr - char to be echoed lb - ptr to led_buf containing cmd_buf.

    EXIT: chr is written to the output que (t_outq) and may be enclosed
    with inverse video escape sequences if the char is non-printable. Tabs
    are expaned to an appropriate number of spaces. Several members in the
    lb struct will have been changed.
*/

static unchar		echo(chr,lb,last)
    register unchar         chr;
    register struct led_buf *lb;
    int			    last;
{
    struct tty_buf          *tbp;
    struct tty              *typ;

    tbp = lb->tbp;
    typ = tbp->ttyp;

    if (lb->current >= last)
	return 0;

    if (CLEKEY_CHAR(chr))
	lb->current += 1,cle_putc(chr,typ);
    else if (chr == '\t')
    {
	chr = 8 - ((lb->current+lb->promptlen) & 7);
	if ((lb->current += chr) >= last)
	{
	    lb->current -= chr;
	    return '\0';
	}

	/*
	    Here we must use spaces and not tabs because we may
	    want to overwrite.
	*/
	while (chr--) cle_putc(' ',typ);

	chr = '\t';
    }
    else
    {
	unchar                  **tcap = (unchar **) tbp->tcap;

	lb->current += 1;

	if (chr == RUB) chr = '?' - '@';
	cle_puts(tcap[TCAP_SETINV],typ);
	cle_putc(chr + '@',typ);
	cle_puts(tcap[TCAP_SETNORM],typ);
    }

    return chr;
}

/*
    Print the prompt string to the terminal

    ENTRY: lb - ptr to led_buf containing prompt string clr_flg - YES if to
    pre-clear the line before printing

    EXIT: prompt string is written to the output que (t_outq). Some chars
    may be converted to displayable ones.
*/

static void		pprompt(lb,clr_flg)
    register struct led_buf *lb;
    int                     clr_flg;
{
    register unchar         *cp;
    register struct tty_buf *tbp = lb->tbp;

    cle_putc('\r',tbp->ttyp);

#if 0
    if (lb->promptlen) cle_puts(lb->prompt,tbp->ttyp);
#else
    for (cp = lb->prompt; *cp && echo(*cp++,lb,tbp->cols-1););
#endif

    if (clr_flg)
	cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);
}

/*
    Print the text to the left of the cursor

    ENTRY: lb - ptr to led_buf containing prompt string clr_flg - YES if to
    pre-clear the line before printing

    EXIT: The prompt string and the text from lb->line[0] through
    lb->line[lb->lcurs] is put in the output que (t_outq). Some characters
    may have bee converted to displayable ones. The message completes with
    an escape sequence to turn on the cursor.
*/

static void             pfront(lb,clr_flg)
    register struct led_buf *lb;
    int                     clr_flg;
{
    register unchar         *cp;

    lb->current = 0;
    pprompt(lb,clr_flg);
    for (cp = lb->line; *cp && echo(*cp++,lb,MAXLINE););
}

/*
    Print the text from the cursor to the end of line

    ENTRY: lb - ptr to led_buf containing prompt string deol - if 1, follows
    message with an escape sequence to delete to end of line. if 0, don't
    delete to end of line

    EXIT: String from lb->line[lb->rcurs] to end of buffer is sent to the
    output que (t_outq).
*/

static void             pback(lb,deol)
    struct led_buf          *lb;
    int                     deol;
{
    register unchar         *cp;
    register int	    tmp1,tmp2;
    unchar                  **tcap;
    struct tty              *typ;
    struct tty_buf          *tbp;
    int			    save;

    tbp = lb->tbp;
    typ = tbp->ttyp;
    tcap = (unchar **) tbp->tcap;

    if (*lb->rcurs == 0)
    {
	if (deol)
	    cle_puts(tcap[TCAP_CLREOL],typ);
	return;
    }

    save = tcap[TCAP_SAVE] && *tcap[TCAP_SAVE];

    if (save)
	cle_puts(tcap[TCAP_SAVE],typ);

    tmp1 = lb->current;
    for (cp = lb->rcurs; *cp && echo(*cp,lb,MAXLINE); cp++);
    tmp2 = lb->lastchar;
    lb->lastchar = lb->current;
    lb->current = tmp1;

    if (save)
    {
	if (deol)
	    cle_puts(tcap[TCAP_CLREOL],typ);
	cle_puts(tcap[TCAP_RESTORE],typ);
    }
    else
    {
	for (tmp1 = tmp2 - lb->lastchar; tmp1--; cle_putc(' ',typ));
	for (tmp1 = tmp2 - lb->current; tmp1--; cle_putc('\b',typ));
    }
}

/*
    Send a message to the terminal

    ENTRY: str - pointer to null terminated string containing message lb -
    ptr to led_buf to which to send the message

    EXIT: message is placed in output que (t_outq) wrapped with the
    necessary escape sequences to make the message appear on the line
    above the current line. Command line is refreshed.
*/

static void             msg(str,lb)
    unchar                  *str;
    struct led_buf          *lb;
{
    long                    oldq;
    register struct tty_buf *tbp;
    register struct tty	    *typ;

    tbp = lb->tbp;
    typ = tbp->ttyp;
    oldq = lb->flags;

    cle_putc('\r',typ);
    cle_puts(tbp->tcap[TCAP_CLREOL],typ);
    cle_puts(tbp->tcap[TCAP_SETINV],typ);
    cle_puts(str,typ);
    cle_puts(tbp->tcap[TCAP_SETNORM],typ);
    cle_putc('\n',typ);
    reprint(lb);
}

/*
    right arrow, move cursor non-destructively
*/
static void             c_r(lb)
    struct led_buf          *lb;
{
    curs_r(1,NODEL,lb);
}

/*
    left arrow, move cursor non-destructively
*/
static void             c_l(lb)
    struct led_buf          *lb;
{
    curs_l(1,NODEL,lb);
}

static void             d_eol(lb)
    struct led_buf          *lb;
{
    register struct tty_buf *tbp = lb->tbp;

    lb->rcurs = lb->lineend;
    *lb->rcurs = '\0';
    cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);
}

static void             d_bol(lb)
    struct led_buf          *lb;
{
    curs_l(MAXLINE,DEL,lb);
}

static void             d_rchr(lb)
    struct led_buf          *lb;
{
    curs_r(1,DEL,lb);
}

static void              d_lchr(lb)
    struct led_buf          *lb;
{
    curs_l(1,DEL,lb);
}

static void              eol(lb)
    struct led_buf          *lb;
{
    curs_r(MAXLINE,NODEL,lb);
}

static void              bol(lb)
    struct led_buf          *lb;
{
    curs_l(MAXLINE,NODEL,lb);
}

static void             tog_insrt(lb)
    struct led_buf          *lb;
{
    lb->flags ^= LD_INSERT;
}

static void             reprint(lb)
    struct led_buf          *lb;
{
    pfront(lb,NO);
    pback(lb,YES);
}

static void             getold_prev(lb)
    register struct led_buf *lb;
{
    register int	    *xptr;
    register struct tty_buf *tbp = lb->tbp;

    xptr = &lb->lastline;
    pprompt(lb,NO);

    lb->lcurs = lb->line;
    lb->rcurs = lb->lineend;
    lb->current = 0;

    /* recall this way eats pattern */
    lb->matchlen = 0;

    if (*ptr(lb->history,*xptr))
	getstr(lb->history,++(*xptr),lb);
    cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);

    lb->flags &=~ (LD_DIRTY);
}

static void             getold_next(lb)
    register struct led_buf *lb;
{
    register int	    *xptr;
    register struct tty_buf *tbp = lb->tbp;

    xptr = &lb->lastline;
    pprompt(lb,NO);

    lb->lcurs = lb->line;
    lb->rcurs = lb->lineend;
    lb->current = 0;

    /* recall this way eats pattern */
    lb->matchlen = 0;
    if (*xptr)
	getstr(lb->history,--(*xptr),lb);
    cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);

    lb->flags &=~ (LD_DIRTY);
}

static void             getold_str(lb)
    register struct led_buf *lb;
{
    register int	    linelen;
    unchar		    *new;
    register struct tty_buf *tbp = lb->tbp;

    eol(lb);

    linelen = (lb->lcurs - lb->line);
    if (linelen == 0)
	return;

    if ((lb->flags&LD_DIRTY) || lb->matchlen == 0)
    {
	lb->matchlen = linelen;
	lb->lastline = 0;
    }
    else
    {
	if (*ptr(lb->history,lb->lastline))
	    lb->lastline++;
	else
	{
	    ring_bell(lb);
	    return;
	}
    }

    new = match_ptr(lb->history,lb->line,lb->matchlen,&lb->lastline,NO);

    if (new == 0)
    {
	ring_bell(lb);
	return;
    }

    pprompt(lb,NO);

    lb->lcurs = lb->line;
    lb->rcurs = lb->lineend;
    lb->current = 0;

    if (linelen = *new++)
	putt(new,--linelen,lb);
    cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);

    lb->flags &=~ (LD_DIRTY);
}

static void		meta(lb)
    struct led_buf	    *lb;
{
    lb->state = ESCAPE;
}

static void		ansi(lb)
    struct led_buf	    *lb;
{
    lb->state = ANSIKEY;
}

/*
    Hardwired ANSI key map for the IBM PC.
*/

static int		ansikey(ch)
    register char	    ch;
{
    switch (ch)
    {
    case '@':	return CLEFUN_INSERT;
    case 'A':	return CLEFUN_PREVIOUS;
    case 'B':	return CLEFUN_NEXT;
    case 'C':	return CLEFUN_CURSR;
    case 'D':	return CLEFUN_CURSL;
    case 'H':	return CLEFUN_GOTOBOL;
    case 'U':	return CLEFUN_SKIPWR;
    case 'V':	return CLEFUN_SKIPWL;
    case 'Y':	return CLEFUN_GOTOEOL;
    default:	return CLEFUN_BELL;
    }
}

static void             superquote(lb)
    struct led_buf          *lb;
{
    putt("^",1,lb); c_l(lb);
    lb->state = QUOTE;
}

static void             self_insert(lb)
    struct led_buf          *lb;
{
    putt(&lb->c,1,lb);
}

static void             nop(lb)
    struct led_buf          *lb;
{
}

static void             ring_bell(lb)
    struct led_buf          *lb;
{
    register struct tty_buf *tbp = lb->tbp;

    cle_puts(tbp->tcap[TCAP_FLASH],tbp->ttyp);
}

/* DON'T CHANGE THE ORDER OF THE ENTRIES IN THE FOLLOWING 3 ARRAYS */

static void             (*edit_functions[]) () =
{
    self_insert,		/* insert a char			*/
    tog_insrt,			/* toggle insert/overstrike mode	*/
    bol,			/* goto bol				*/
    eol,			/* goto eol				*/
    d_lwrd,			/* delete "word" to the left		*/
    d_rwrd,			/* delete "word" to the right		*/
    d_bol,			/* delete to beginning of line		*/
    d_eol,			/* delete to end of line		*/
    c_l,			/* cursor left				*/
    c_r,			/* cursor right				*/
    d_lchr,			/* delete character on left		*/
    d_rchr,			/* delete character on right		*/
    reprint,			/* refresh the line			*/
    getold_prev,		/* get previous				*/
    getold_next,		/* get next				*/
    getold_str,			/* get matching string			*/
    newline,			/* end of input				*/
    superquote,			/* escape next char			*/
    nop,			/* do nothing				*/
    ring_bell,			/* echo a bell				*/
    skipwl,			/* skip word right			*/
    skipwr,			/* skip word left			*/
    nop,			/* actually purge (trapped at interrupt time) */
    meta,			/* meta/ESC key				*/
    ansi,			/* ANSI cursors, ESC [			*/
    0				/* trailer				*/
};

static unchar           hex[] = "0123456789ABCDEF";

static unchar           *itoa(val,ptr,radix)
    register unsigned int   val;
    unchar                  *ptr;
    register int	    radix;
{
    unsigned		    quo;
    register unsigned	    rem;

    rem = val % radix;
    quo = val / radix;
    if (quo != 0)
	ptr = itoa(quo,ptr,radix);
    *ptr++ = hex[rem];
    return ptr;
}

/*
    parse_it is the entry point for the editor. It is called for each
    character input from the keyboard. The process context is task time. It
    must never be called from an interrupt routine.

    ENTRY: lb - ptr to led_buf for this process. lb->c has character to
    process.

    EXIT: the command line is edited.
*/

static void             parse_it(lb)
    register struct led_buf *lb;
{
    int			    s = lb->state;
    register unchar	    c = lb->c;
    register unchar	    *kmap = lb->tbp->keymap;

    lb->state = NORMAL; /* next default state ... */

    switch (s)
    {
    case NORMAL:
	if (CLEKEY_CHAR(c))
		putt(&lb->c,1,lb);
	else	(*edit_functions[kmap[CLEKEY_CMD(c)]]) (lb);
	break;

    case ESCAPE:
	if (CLEKEY_CHAR(c))
		(*edit_functions[kmap[c]]) (lb);
	else    ring_bell(lb);
	break;

    case ANSIKEY:
	(*edit_functions[ansikey(c)]) (lb);
	break;

    case QUOTE:
	d_rchr(lb);
	if (c != '\0')
		putt(&lb->c,1,lb);
	else    msg("NULs cannot be inserted",lb);
	break;

    default:

	switch (s)
	{
	case NCC_SPC(VEOL):	newline(lb);	break;
	case NCC_SPC(VERASE):	d_lchr(lb);	break;
	case NCC_SPC(VKILL):	d_bol(lb);	break;

	default:
	    msg("Unknown key and/or key state",lb);
	}
    }
}

/*
    The following n routines are specific to the line discipline and are what
    are called by the kernel. Process context can be both task time and
    interrupt time and is so indicated at the entry point of each routine.

    The pre-processor variable INKERNEL is defined when compiling for line
    discipline mode. If it is not defined, then the program will be made in
    standalone mode which is useful for debugging the edit only portion of the
    program.
*/

/*
    allocate a dummy u struct if s.a. mode
*/
#if !INKERNEL
struct user
{
    unchar                  *u_base;
    int                     u_count;
    int                     u_error;
    struct proc             *u_procp;
}                       u;
#endif

#if MULTILB
static struct led_buf	*ldb_free;
#endif

static struct tty_buf	*tty_free;
static struct tty_buf	*tty_used;

#if DEBUG
static VOID             dump_ttybuf(tbp,msg)
    struct tty_buf          *tbp;
    unchar                  *msg;
{
    register struct tty_buf *nxt;

    if (tbp == 0)
    {
	printf("%s: tbp = 0\n",msg);
	return;
    }

    nxt = tbp;

    do
    {
	printf("%s: tbp = %X, next = %X, last = %X, tp = %X\n\tlbp = %X, flags = %X\n",
	    msg,nxt,nxt->next,nxt->last,nxt->ttyp,nxt->lbp,nxt->flags);
	nxt = nxt->next;
    }
    while (nxt != tbp);

}

static void             dump_ledbuf(lb,msg)
    struct led_buf          *lb;
    unchar                  *msg;
{
    struct led_buf          *nlb;

    if (lb == 0)
    {
	printf("%s: lb = 0\n",msg);
	return;
    }

    nlb = lb;
    do
    {
	printf("%s: lb = %X, next = %X, last = %X, td = %X\n\tttybf = %X, flags = %X\n",
	    msg,nlb,nlb->next,nlb->last,nlb->tbp->ttyp,nlb->tbp,nlb->flags);
	nlb = nlb->next;
    }
    while (nlb != lb);
}

#if MULTILB
static void             dump_buftree(tbp)
    struct tty_buf          *tbp;
{
    struct led_buf          *lb;
    struct tty_buf          *utbp;

    printf("Into dump_buftree(): tbp = %X\n",tbp);
    dump_ttybuf(tty_used,"Used");
    dump_ttybuf(tty_free,"Free");

    if (tbp != 0)
    {
	printf("lbp = %X\n",tbp->lbp);
	dump_ledbuf(tbp->lbp,"Used");
    }

    dump_ledbuf(ldb_free,"Free");
    printf("Strike any key to continue: ");
    (void) getchar();
    printf("Out of dump_buftree()\n");
}
#endif

static void             dump_clist(clp,msg)
    struct clist            *clp;
    unchar                  *msg;
{
    struct cblock           *cbp;
    int                     size;

    printf("dump_clist() [%s]: ptr = %x\n",msg,clp);

    if (clp == 0 || clp->c_cf == 0)
    {
	printf("clist is empty\n");
	return;
    }

    printf("\tcc = %d, first = %x, last = %x\n",clp->c_cc,clp->c_cf,clp->c_cl);
    cbp = clp->c_cf;

    size = 0;
    do
    {
	unchar                  tstr[clsize + 1],*dst,*src;
	int                     i;

	dst = tstr;
	src = &cbp->c_data[(i = cbp->c_first)];
	for (; i < cbp->c_last; ++i)
	{
	    unchar                  c;

	    c = *src++;
	    if (!CLEKEY_CHAR(c))
		c = '.';
	    *dst++ = c;
	}
	*dst = '\0';

	printf("\t%x, next = %x, first = %d, last = %d, size = %d\n",
	    cbp,cbp->c_next,cbp->c_first,cbp->c_last,cbp->c_last - cbp->c_first);
	printf("\tstr = {%s}\n",tstr);

	size += cbp->c_last - cbp->c_first;
	if (cbp == clp->c_cl)
	    break;

	cbp = cbp->c_next;
    }
    while (cbp != 0);

    if (size != clp->c_cc)
	printf("\taccumulated size of %d doesn't match c_cc size of %d\n",
	    size,clp->c_cc);

    printf("type any char to continue");
    (void) getchar();
    putchar('\n');
}
#endif /* DEBUG */

/*
    set the keymap to defaults.
*/

static void             setup_key_defaults(tbp)
    struct tty_buf          *tbp;
{
    register int            cnt;
    register unchar	    *kmap;

    for (kmap = tbp->keymap, cnt = 0; cnt < CLEKEY_MAX; ++cnt)
	*kmap++ = CLEFUN_BELL;

    kmap = tbp->keymap;

    kmap['\0']			= CLEFUN_NOP;
    kmap[CLEKEY_CTL('A')]	= CLEFUN_GOTOBOL;
    kmap[CLEKEY_CTL('B')]	= CLEFUN_CURSL;
    kmap[CLEKEY_CTL('D')]	= CLEFUN_DELCRIT;
    kmap[CLEKEY_CTL('E')]	= CLEFUN_GOTOEOL;
    kmap[CLEKEY_CTL('F')]	= CLEFUN_CURSR;
    kmap[CLEKEY_CTL('G')]	= CLEFUN_BELL;
    kmap[CLEKEY_CTL('H')]	= CLEFUN_DELCRIT;
    kmap[CLEKEY_CTL('I')]	= CLEFUN_CHAR;
    kmap['\n']			= CLEFUN_NEWLINE;
    kmap[CLEKEY_CTL('K')]	= CLEFUN_DELEOL;
    kmap[CLEKEY_CTL('L')]	= CLEFUN_REFRESH;
    kmap['\r']			= CLEFUN_NEWLINE;
    kmap[CLEKEY_CTL('N')]	= CLEFUN_NEXT;
    kmap[CLEKEY_CTL('P')]	= CLEFUN_PREVIOUS;
    kmap[CLEKEY_CTL('Q')]	= CLEFUN_ESCAPE;
    kmap[CLEKEY_CTL('R')]	= CLEFUN_FIND;
    kmap[CLEKEY_CTL('U')]	= CLEFUN_DELBOL;
    kmap[CLEKEY_CTL('V')]	= CLEFUN_ESCAPE;
    kmap[CLEKEY_CTL('W')]	= CLEFUN_DELWLFT;
    kmap[CLEKEY_CTL('Y')]	= CLEFUN_PURGE;
    kmap[CLEKEY_CTL('^')]	= CLEFUN_ESCAPE;
    kmap[ESC]			= CLEFUN_META;
    kmap[CLEKEY_CTL('\\')]	= CLEFUN_FIND;
    kmap[RUB]			= CLEFUN_DELCLFT;

    kmap[CLEKEY_ESC('d')]	= CLEFUN_DELWRIT;
    kmap[CLEKEY_ESC('f')]	= CLEFUN_SKIPWR;
    kmap[CLEKEY_ESC('b')]	= CLEFUN_SKIPWL;
    kmap[CLEKEY_ESC('p')]	= CLEFUN_FIND;
    kmap[CLEKEY_ESC('[')]	= CLEFUN_ANSI;
    kmap[CLEKEY_ESC('-')]	= CLEFUN_DELWLFT;
}

/*
    The following init's setup the strings required to make a terminal do the
    given functions.
*/

static void             setup_tcap_defaults(tbp)
    struct tty_buf          *tbp;
{
    tbp->tcap[TCAP_CLREOL]	= TCAP_CLREOL_STR;
    tbp->tcap[TCAP_SETINV]	= TCAP_SETINV_STR;
    tbp->tcap[TCAP_SETNORM]	= TCAP_SETNORM_STR;
    tbp->tcap[TCAP_SAVE]	= TCAP_SAVE_STR;
    tbp->tcap[TCAP_RESTORE]	= TCAP_RESTORE_STR;
    tbp->tcap[TCAP_FLASH]	= TCAP_FLASH_STR;

#if M_SPTALLOC
    if (tbp->tcstrings != 0)
    {
	sptfree(tbp->tcstrings,tbp->tclen,1);
	tbp->tcstrings = 0;
	tbp->tclen = 0;
    }
#else
    tbp->tclen = 0;
#endif
}

/*
    Initialize the tty and led free lists. Threads the entries in the
    tty buf array together and on the tty_free list, and either
    assigns each led buf to a tty buf or threads the led buf array in
    a free list, depending on whether MULTILB is true.
*/

static void		freelist_init()
{
    register int	    cnt;

#if MULTILB
    {
	register struct led_buf	*lastlb;

	/*
	    Thread first led buf
	*/
	lastlb = &cle_buffers[0];
	lastlb->next = lastlb->last = lastlb;

	/*
	    Append the rest, if any, to the last
	*/
	for (lastlb, cnt = 1; cnt < cle_ttys; cnt++,lastlb++)
	{
	    register struct led_buf *lb = lastlb+1;

	    lb->next = lastlb->next;
	    lb->last = lastlb;
	    lb->next->last = lastlb->next = lb;
	}

	ldb_free = cle_buffers;
    }
#endif

    {
	register struct tty_buf *lasttbp;

	/*
	    Thread first tty buf
	*/
	lasttbp = cle_ttybuf;
	lasttbp->next = lasttbp->last = lasttbp;
	lasttbp->lbp = 0;

	/*
	    Append rest, if any, to the end
	*/
	for (lasttbp, cnt = 1; cnt < cle_ttys; cnt++, lasttbp++)
	{
	    register struct tty_buf *tbp = lasttbp+1;

	    tbp->lbp = 0;
	    tbp->next = lasttbp->next;
	    tbp->last = lasttbp;
	    tbp->next->last = lasttbp->next = tbp;
 	}
    }

#if !MULTILB
    {
	register struct led_buf *lb = &cle_buffers[0];
	register struct tty_buf *tbp = &cle_ttybuf[0];

	for (lb, tbp, cnt = 0; cnt < cle_ttys; cnt++, lb++, tbp++)
	{
	    lb->tbp = tbp;
	    tbp->lbp = lb;
	}
    }
#endif

    tty_free = cle_ttybuf;
}

static void		zap_lb(lb,flags)
    register struct led_buf *lb;
    int			    flags;
{
    lb->flags = (flags&TB_INSERT) ? LD_INSERT : 0;

    /*
	Last byte is a fake key definition buffer, length 1 byte, empty.
    */

    *lb->history	= 1;
    *(lb->history+1)	= '\0';

    lb->historyend	= lb->history+MAXHISTORY - 1;
    *lb->historyend	= '\0';

    lb->owed = 0;

    lb->promptlen = 0;
}

#if INKERNEL

#if MULTILB
/*
    Put the led buf beloning to the given tty buf back on the free list.
*/

static struct led_buf	*free_lb(lb,tbp)
    register struct led_buf *lb;
    register struct tty_buf *tbp;
{
    struct led_buf	    *nxtlb;

    /*
	Detach from the list of led bufs for this tty buf
    */

    if (lb == lb->next)
	nxtlb = tbp->lbp = 0;
    else
    {
	nxtlb = lb->next;
	lb->next->last = lb->last;
	lb->last->next = lb->next;
	if (lb == tbp->lbp)
	    tbp->lbp = lb->next;
    }

    /*
	Clean things up
    */

    lb->proc = 0;
    lb->flags = 0;
    lb->tbp = 0;

    /*
	Attach to free list
    */

    if (ldb_free == 0)
	lb->next = lb->last = lb;
    else
    {
	lb->last = ldb_free->last;
	lb->next = ldb_free;
	lb->last->next = ldb_free->last = lb;
    }

    ldb_free = lb;

    return nxtlb;
}

/*
    For a given tty buffer, scan the chain of led bufs and free any that
    are attached to defunct processes.
*/

static void		free_leds(tbp)
    register struct tty_buf *tbp;
{
    register struct led_buf *lb;

    lb = tbp->lbp;

    do
    {
	lb = 
	(
	    lb->tbp == 0
	    || lb->proc == 0
	    || lb->proc->p_pid != lb->pid
	    || lb->proc->p_ppid != lb->ppid
	)
	    ? free_lb(lb,tbp) : lb->next;
    }
    while (lb != tbp->lbp);
}
#endif /* MULTILB */

static struct tty_buf	*free_ttybuf(tbp)
    register struct tty_buf *tbp;
{
    struct tty_buf	    *nxttbp;

    /*
	Detach it from the tty_used list
    */

    if (tbp == tbp->next)
	nxttbp = tty_used = 0;
    else
    {
	int                     ospl;

	ospl = spl6();
	{
	    tbp->next->last = tbp->last;
	    tbp->last->next = tbp->next;
	    if (tbp == tty_used)
		tty_used = tbp->next;
	    nxttbp = tbp->next;
	}
	splx(ospl);
    }

    /*
	Disgorge any pending output, and clean things up
    */

    if (tbp->broadcast.c_cf != 0)
    {
	struct cblock           *cnxt;
	int                     ospl;

	ospl = spl6();
	{
	    while ((cnxt = getcb(&tbp->broadcast)) != 0)
		putcf(cnxt);
	    kick_out(tbp->ttyp);
	}
	splx(ospl);
    }

    tbp->flags = 0;
    tbp->dorefresh = tbp->readsleep = NO;

    /*
	Attach to the free list
    */

    if (tty_free == 0)
	tbp->next = tbp->last = tbp;
    else
    {
	register struct tty_buf	*tn;

	tn = tty_free->next;
	(tbp->next = tn)->last = (tbp->last = tty_free)->next = tbp;
    }

    tty_free = tbp;

    return nxttbp;
}

/*
    For each tty buf, free up all the led bufs not attached to
    any process, and free all those (except for the one passed
    as an argument) that have no led bufs left and are not
    open.
*/

static void             free_ttys(oldtbp)
    struct tty_buf          *oldtbp;
{
    register struct tty_buf *tbp;

    tbp = tty_used;

    do
    {
	register struct tty_buf *nxttbp = tbp->next;

#if MULTILB
	if (tbp->lbp != 0)
	    free_leds(tbp);
#endif

	tbp =
	(
	    tbp != oldtbp
	    && tbp->readsleep == NO
#if MULTILB
	    && !(tbp->flags&TB_OPENING)
	    && tbp->lbp == 0
#else
	    && !(tbp->flags&TB_OPEN)
#endif
	)
	    ? free_ttybuf(tbp) : tbp->next;
    }
    while (tbp != tty_used);
}
#endif /* INKERNEL */

/*
    Ran out of tty_buf's or led_buf's so ripple through the allocated ones and
    deallocate any that are no longer being used. It is also called to init
    the links in the structures.

    ENTRY: Process context: task. NEVER call from an interrupt routine.
    oldtbp: ptr to tbp buffer of buffer not to put on free list; tty_free:
    points to first element in free list for tty_buf's; tty_used: points to
    first element in used list for tty_buf's; ldb_free: points to first element
    in free list for led_buf's

    EXIT: all led_buff's assigned to defunct processes are placed back in
    the free list. All tty_buf's that have no led_buf's assigned are
    placed back in the free list.
*/

static void             garbage_collect(oldtbp)
    struct tty_buf          *oldtbp;
{
    if (tty_used == 0 && tty_free == 0)
	freelist_init();
#if INKERNEL
    else if (tty_used != 0)
	free_ttys(oldtbp);
#endif
}

/*
    Get the next available led_buf from the freelist.

    ENTRY: Process context: task. Must never be called from
    interrupt routine. tbp - ptr to tty_buf assigned to tty
    struct to which led_buf is to be attached.

    EXIT: returns ptr to led_buf if one is available else returns 0. If
    led_buf is attached, it is placed at the head of the que; tbp->lbp
    will be moved in that case. The led_buf is initialised for use.  The
    history buffer is purged.
*/

static struct led_buf   *get_ledbuf(tbp)
    struct tty_buf          *tbp;
{
    struct led_buf          *lb,*next,*last;
    int                     cnt;
    unchar                  *chr;

#if !MULTILB
    lb = tbp->lbp;
#else
    if (ldb_free == 0)
    {
	garbage_collect(tbp);
	if (ldb_free == 0)
	    return 0;
    }

    /*
	Detach first led buf in free chain
    */

    lb = ldb_free;

    if (lb->next == lb)
	ldb_free = 0;
    else
    {
	lb->next->last = lb->last;
	lb->last->next = lb->next;
	ldb_free = lb->next;
    }

    /*
	Attach as first led buf in the tty buf' chain.
    */

    if (tbp->lbp == 0)
	lb->next = lb->last = lb;
    else
    {
	next = tbp->lbp;
	last = next->last;
	(lb->last = last)->next = (lb->next = next)->last = lb;
    }

#if INKERNEL
    lb->proc	= u.u_procp;
    lb->pid	= u.u_procp->p_pid;
    lb->ppid	= u.u_procp->p_ppid;
#endif

    lb->tbp = tbp;
    tbp->lbp = lb;
#endif /* MULTILB */

    zap_lb(lb,tbp->flags);

    return lb;
}

#if MULTILB
/*
    Find the led_buf in the list belonging to this process.

    ENTRY: Process context: task. Must never be called from interrupt
    routine. tbp - ptr to tty_buf assigned to tty struct which has led_buf que
    uproc - pointer to element in process table (used only to inspect the
    p_pid and p_ppid fields).

    EXIT: returns ptr to led_buf if one is found else returns 0. If
    led_buf is found, it is placed at the head of the que; tbp->lbp may
    be moved in that case.
*/

static struct led_buf   *find_ledbuf(tbp,uproc)
    struct tty_buf          *tbp;
    struct proc             *uproc;
{
    register struct led_buf *lb;
    int                     cnt = 0;

    if (tbp->lbp == 0)
	return 0;

    lb = tbp->lbp;

    do
    {
#if INKERNEL
	if ((lb->proc != 0 && lb->proc == uproc)
	    && (lb->pid == uproc->p_pid && lb->ppid == uproc->p_ppid))
	{
	    if (lb != tbp->lbp)
	    {
		struct led_buf          *next,*last;

		lb->last->next = lb->next;
		lb->next->last = lb->last;
		next = tbp->lbp;
		last = next->last;
		(lb->last = last)->next = (lb->next = next)->last = lb;
		tbp->lbp = lb;
	    }

	    return lb;
	}
#endif /* INKERNEL */

	lb = lb->next;
	cnt++;
    }
    while (lb != tbp->lbp && cnt < cle_leds);

    return 0;
}

#endif /* MULTILB */

static void		setcols(tbp,cols)
    register struct tty_buf *tbp;
    int			    cols;
{
    if (cols < 8)
    {
	u.u_error = ERR_BADPARAM;
	return;
    }

    if (cols >= MAXLINE)
	cols = MAXLINE;

    tbp->cols   = cols;
}

/*
    Get a tty_buf from the free pool.
*/

/*
    ENTRY: Process context: task. Never call this from an interrupt
    routine. tp - ptr to tty struct to which the tty_buf will become
    associated.

   
    EXIT: returns ptr to tty_buf or 0 if there aren't any available.
    tty_buf is removed from the free list and inserted into the used list
    which may make tty_used and tty_free change values.
*/

static struct tty_buf   *get_ttybuf(tp)
    struct tty              *tp;
{
    struct tty_buf          *tbp,*next,*last;
    int                     cnt;
    unchar                  *chr;

    if (tty_free == 0)
    {
	garbage_collect(tbp);
	if (tty_free == 0)
	    return 0;
    }

    tbp = tty_free;

    if (tbp->next == tbp)
	tty_free = 0;
    else
    {
	tbp->next->last = tbp->last;
	tbp->last->next = tbp->next;
	tty_free = tbp->next;
    }

    tbp->ttyp = tp;
    setcols(tbp,COLUMNS);
    tbp->dorefresh = tbp->readsleep = NO;

#if !MULTILB
    get_ledbuf(tbp);
#else
    tbp->lbp = 0;
#endif

    next = tty_used;
    if (next == 0)
	tbp->next = tbp->last = tbp;
#if INKERNEL
    else
    {
	int                     ospl;

	ospl = spl6();
	{
	    last = next->last;
	    (tbp->last = last)->next = (tbp->next = next)->last = tbp;
	}
	splx(ospl);
    }
#endif

    tbp->flags = TB_INSERT;
    setup_tcap_defaults(tbp);
    setup_key_defaults(tbp);

    tty_used = tbp;
    return tbp;
}

/*
    Find a tty_buf that's attached to a tty struct and move it to the head of
    the que if it's in the list.

    ENTRY: Process context: task. Must not be called from interrupt. tp -
    ptr to tty struct to match

    EXIT: returns ptr to tty_buf or 0 if none found. Does not modify the
    contents of any structures, so it is re-entrant and interruptable.
*/

static struct tty_buf   *find_ttybuf(tp)
    struct tty              *tp;
{
    register struct tty_buf *tbp;
    int                     cnt = 0;

    if (tty_used == 0)
	return 0;

    tbp = tty_used;

    do
    {
	if (tbp->ttyp == tp)
	{
#if INKERNEL
	    if (tbp != tty_used)
	    {
		int                     ospl;

		ospl = spl6();
		{
		    register struct tty_buf *next,*last;

		    tbp->next->last = tbp->last;
		    tbp->last->next = tbp->next;
		    next = tty_used;
		    last = next->last;
		    (tbp->last = last)->next = (tbp->next = next)->last = tbp;
		    tty_used = tbp;
		}
		splx(ospl);
	    }
#endif /* INKERNEL */
	    return tbp;
	}

	tbp = tbp->next;
	cnt++;
    }
    while (tbp != tty_used && cnt < cle_ttys);

    return 0;
}

/*
    The routines cleopen, cleclose, cleread, clewrite and cleioctl are called
    by the kernel (via a dispatch through the linesw array) and are all
    executed at process task time. The routines cleinput and cleouput are
    interrupt routines. Except for cleinput and cleioctl all the routines have
    the same interface: a pointer to the tty struct assigned to the real or
    pseudo terminal. cleinput has an extra flag argument which indicates
    whether the input character is real or the result of the terminal sending
    a break. cleioctl has an extra argument that has the function code the
    ioctl is to do.
*/

/*
    Send saved broadcast message to terminal.

    ENTRY: Process context: task. tbp - ptr to tty_buf which contains the
    message clr_flg - true if the message line is to be precleared

    EXIT:  the cblock(s) holding the message are moved from the clist in
    tty_buf to the t_outq clist.
*/

static void		send_brdcst(tbp,clr_flg)
    struct tty_buf          *tbp;
    int                     clr_flg;
{
#if INKERNEL
    int                     ospl;
    unchar                  c;
    struct cblock           *bp;
    struct clist            *bl,*tpc;

    tpc = &tbp->ttyp->t_outq;
    bl = &tbp->broadcast;
    if (clr_flg)
    {
	cle_putc('\r',tbp->ttyp);
    	cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);
    }

    ospl = spl6();
    {
	while ((bp = getcb(bl)) != 0)
	    putcb(bp,tpc);
	kick_out(tbp->ttyp);
    }
    splx(ospl);
#endif /* INKERNEL */
}

/*
    Open line discpline

    ENTRY: tbp - ptr to tty_buf to open (or 0 if not preassigned) td - ptr
    to tty struct  to open
*/

static struct tty_buf   *open_it(tbp,td)
    register struct tty_buf *tbp;
    struct tty              *td;
{
    int                     wasclosed;

#if INKERNEL
    if (tbp != 0)
	wasclosed = !(tbp->flags&TB_OPEN);
    else
    {
	garbage_collect();
	tbp = get_ttybuf(td);
	wasclosed = YES;
    }

    if (tbp != 0)
    {
	tbp->flags |= (TB_OPEN|TB_OPENING);
	if (wasclosed)
	{
#if VERBOSE
	    show_pid(1," started by process ",td);
#endif
	}
    }
    else
    {
	if (wasclosed)
	{
	    cle_puts("\rNo buffers available to start cled\r\n",td);
	    u.u_error = ERR_NOTTYBUF;
	}
    }

    if (wasclosed)
	kick_out(td);
#endif

    return tbp;
}

/*
    Open the line discipline. This procedure is called by kernel code when the
    line discipline is selected. I haven't yet determined what exactly the
    open is supposed to do, but since cled uses discipline 0, if IT (ld0)
    isn't opened too, very strange terminal behavior results.

    ENTRY: tp - ptr to tty struct process context: task.

    EXIT:  discipline 0 is opened.
*/

cleopen(tp)
    struct tty              *tp;
{
#if INKERNEL
    (*linesw[0].l_open) (tp);
    (void) open_it(find_ttybuf(tp),tp);
#endif

    return 1;
}

/*
    Close the line discpline

    ENTRY: tbp - ptr to tty_buf to close (or 0 if not preassigned) td - ptr
    to tty struct  to open
*/

static void             close_it(tbp,td)
    struct tty_buf          *tbp;
    struct tty              *td;
{
#if INKERNEL
    if (tbp != 0)
    {
	int                     ospl;

	if (td->t_state&ISOPEN)
	{
#if VERBOSE
	    show_pid(0,"\rcled stopped by process ",td);
#endif
	    kick_out(td);
	}

	if (tbp->readsleep)
	{
	    wakeup(tbp);
	    tbp->readsleep = NO;
	    if (td->t_state&ISOPEN)
		tbp->flags |= TB_FLUSHIT;
	}

	ospl = spl6();
	{
	    tbp->flags &=~ (TB_OPEN|TB_OPENING);
	}
	splx(ospl);
    }

#if MULTILB
    free_leds(tbp);
#endif

#endif
}

/*
    Close the line discipline. This procedure is called by kernel code when
    the line discipline is deselected. I haven't yet determined what exactly
    the close is supposed to do, but a call to close discipline 0 is done
    because it apparently needs it.

    ENTRY: tp - ptr to tty struct process context: task.

    EXIT:  discipline 0 is closed.
*/

cleclose(tp)
    struct tty              *tp;
{
    struct tty_buf	    *tbp;

    tbp = find_ttybuf(tp);

#if INKERNEL
    (*tp->t_proc) (tp,T_RESUME);
    close_it(tbp,tp);
    delay(HZ);
    ttyflush(tp,FWRITE|FREAD);
    (*linesw[0].l_close) (tp);
#endif

    return 1;
}

#if INKERNEL

/*
    Input interrupt routine. This routine is called after n characters have
    been placed in the interrupt holding buffer (t_rbuf) and/or a certain
    time has elapsed since the last input interrupt occurred (This technique
    tends to reduce the CPU load somewhat by bunching input character
    processing on dumb terminal ports).

    The input routine processes the characters in the device buffer, which
    is an array, and then appends them to the raw queue, which is a clist.
    It wakes up the waiting process, which will eventually call cleread that
    will get the characters from the raw queue, to its internal history, and
    then to the user.

    If the line discipline is in raw mode, we check for VINTR and VQUIT and
    send the appropriate signal. Notice that we cannot do this in cleread,
    because cleread is only called when the user reads. We want to be able
    to interrupt even if no read is outstanding.
*/

cleinput(td,bflg)
    struct tty              *td;
    int                     bflg;
{
    int                     i,indx,ospl;
    unchar		    ch,*cp,*msg;
    struct tty_buf          *tbp;

    i = 0;
    tbp = tty_used;

    if (tbp != 0)
	do
	{
	    if (tbp->ttyp == td)
		break;
	    tbp = tbp->next;
	    ++i;
	}
	while ((tbp != tty_used) && i < cle_ttys);

    /*
	If we cannot run, let ldisc 0 sort it out...
    */

    if (CLEDOFF(td->t_lflag) || tbp == 0 || tbp->ttyp != td
	|| (tbp->ttyp == td && (tbp->flags&TB_NOLINE)))
    {
	(*linesw[0].l_input) (td,bflg);
	return;
    }

    /*
	Our input is either a break condition or rbuf.
    */

    if (bflg == L_BREAK)
    {
	if (!((td->t_lflag&ISIG) && (td->t_iflag&BRKINT)))
	    return;

	cp = &ch;
	ch = td->t_cc[VINTR];
	i = 1;
    }
    else
    {
	if (td->t_rbuf.c_ptr == 0 || td->t_rbuf.c_count >= td->t_rbuf.c_size)
	    return;

	i = td->t_rbuf.c_size - td->t_rbuf.c_count;
	cp = (unchar *) td->t_rbuf.c_ptr;
	td->t_rbuf.c_count = td->t_rbuf.c_size;
    }

    /*
	Here we have cp that points at the array of chars to process,
	and i is the number of such characters.
    */

    for (i, cp; i > 0; --i, cp++)
    {
	register unchar		c;
	int                     quote;

	c = *cp;

	if (c == '\0')
	    continue;

	/*
	    The switch character is very special. We cannot even
	    escape it.
	*/
	if (c == td->t_cc[VSWTCH])
	{
	    if ((*td->t_proc) (td,T_SWTCH) != T_SWTCH)
		return;
	}

	if (quote = (td->t_state & CLESC))
	    td->t_state &=~ CLESC;

	if (!quote)
	{
	    unchar			quit,intr;

	    quit = td->t_cc[VQUIT];
	    intr = td->t_cc[VINTR];

	    if (c == quit || c == intr)
	    {
		if (!(td->t_lflag&NOFLSH))
		    ttyflush(td,FWRITE|FREAD);

		kick_out(td);

		tbp->lbp->flags |= (c == intr) ? LD_INTR : LD_QUIT;

		if (tbp->readsleep)
		{
		    tbp->flags |= TB_FLUSHIT;
		    tbp->readsleep = NO;
		    wakeup(tbp);
		}

		signal(td->t_pgrp,(c == intr) ? SIGINT : SIGQUIT);

		return;
	    }
	    else if (!CLEKEY_CHAR(c)
		&& tbp->keymap[CLEKEY_CMD(c)] == CLEFUN_PURGE)
	    {
		ttyflush(td,FREAD);
		if (tbp->readsleep)
		{
		    tbp->flags |= TB_FLUSHIT;
		    break;
		}
	    }
	}

	if (td->t_rawq.c_cc > (TTYHOG-3) || putc(c,&td->t_rawq) == -1)
	{
	    tbp->flags |= TB_OVERUN;
	    cle_putc(BEL,td);
	    continue;
	}

	if (!quote)
	{
	    if (!CLEKEY_CHAR(c) && tbp->keymap[CLEKEY_CMD(c)] == CLEFUN_ESCAPE)
		td->t_state |= CLESC;
	    else if
	    (
		c == td->t_cc[VEOL]
		|| c == td->t_cc[VEOL2]
		|| c == td->t_cc[VEOF]
		|| (!CLEKEY_CHAR(c)
		    && tbp->keymap[CLEKEY_CMD(c)] == CLEFUN_NEWLINE))
	    {
		/* count it so rdchk() works */
		td->t_delct++;
	    }
	}
    }

    if (tbp->readsleep)
    {
	tbp->readsleep = NO;
	wakeup(tbp);
    }

#if M_UNIX
    /*
       if by some chance, we're turned on while LD 0 is reading, wake it up
       too
    */
    if (td->t_state & IASLP)
#if M_ATT
	ttrstrt(td);
#else
	ttiwake(td);
#endif
#endif
}

/*
    Output interrupt routine. This routine does nothing at this time. Control
    is passed to discipline 0 for output. It normally just moves text from the
    t_outq clist to the t_obuf output que. Null chars and timers may be set
    appropriately depending on the char being output. This discipline has no
    interest in changing the behaviour of the output routines.

    ENTRY: td - ptr to tty struct Process context: Interrupt.

    EXIT:  Characters may have been moved from t_outq to t_obuf and the
    output started.
*/

cleoutput(td)
    struct tty              *td;
{
    return (*linesw[0].l_output) (td);
}

/*
    read timeout routine. This is a a kludge cuz I can't figure out who's
    zapping the tty struct out from under cled. It usually happens after the
    cleread routine has gone to sleep. It can be forced, however by some other
    process intentionally changing the line number or other tty flags while a
    read is in progress. This routine helps correct the situation.

    ENTRY: arg - ptr to the tty_buf belonging to the process doing the read.

    EXIT: TB_TIMEOUT set in the tty flags and a wakeup is delivered.
*/

static void             read_timeout(arg)
    struct tty_buf          *arg;
{
    if (arg->readsleep)
	wakeup(arg);
    arg->readsleep = NO;
}

/*
    Announce that's there's no room at the inn

    ENTRY: tp - ptr to tty struct typ - ascii string describing buffer type
    that we're out of

    EXIT: message output to the terminal line discipline reset to ld 0.
*/

static int		no_room(tp,typ)
    struct tty              *tp;
    unchar                  *typ;
{
    cle_puts("\rInsufficient ",tp);
    cle_puts(typ,tp);
    cle_puts(" buffers to run cled at this time.\r",tp);
    tp->t_line = 0;
    (*linesw[0].l_write) (tp);
    return 1;
}

#endif /* INKERNEL */

/*
    Read a line from the terminal. This routine exits when the user types the
    specified eol character (normally a \r).

    ENTRY: Process context: task. tp - ptr to tty struct

    EXIT: process sleeps until user types one of an EOL, an EOF, a QUIT or
    an INTR character (normally an \r, ^D, DEL and ^C characters
    respectively). The line of text input from terminal is moved to user's
    buffer as specified by u.u_count and u.u_base. An EOF results in 0
    chars being returned. This routine will not exit normally if an
    interrupt or quit character is typed (the kernel takes control via the
    signal and will cancel the I/O request without this routine going
    through its normal exit sequence). If the terminal has been set to
    "raw" mode, this read passes control to discipline 0, otherwise the
    line editing functions of cled become active while the line of input is
    being typed.
*/

cleread(tp)
    struct tty              *tp;
{
    struct led_buf          *lb;
    struct tty_buf          *tbp;
    unchar                  *cp;
    int                     c,len,time_id;

    tbp = find_ttybuf(tp);

#if INKERNEL
    if (tbp == 0)
	return no_room(tp,"tty");

    if (u.u_count == 0 || CLEDOFF(tp->t_lflag))
    {
	if (tbp != 0)
	    tbp->flags |= TB_NOLINE;
	(*linesw[0].l_read) (tp);
	return 1;
    }

    if (!(tbp->flags&TB_OPEN))
	open_it(tbp,tp);
#endif

#if !MULTILB
    lb = tbp->lbp;
#else
    lb = find_ledbuf(tbp,u.u_procp);
    if (lb == 0)
	lb = get_ledbuf(tbp);
#endif

    if (tbp->broadcast.c_cc != 0)
    {
	send_brdcst(tbp,0);
	cle_putc('\n',tp);
	tbp->dorefresh = YES;
    }

#if INKERNEL
    if (lb == 0)
    {
	close_it(tbp,tp);
	return no_room(tp,"led");
    }
#endif

    tbp->flags &=~ (TB_NOLINE|TB_OPENING);

#if INKERNEL
    if (lb->owed != 0)
    {
	len = lb->lcurs - lb->owed;

	if (len > 0)
	{
	    if (len > u.u_count)
		len = u.u_count;
	    if (copyout(lb->owed,u.u_base,len) < 0)
	    {
		u.u_error = EFAULT;
		return 1;
	    }
	    lb->owed += len;
	    u.u_base += len;
	    u.u_count -= len;
	    if (lb->owed >= lb->lcurs)
		lb->owed = 0;
	    return 1;
	}

	lb->owed = 0;
    }
#endif

    /*
	Initialize command buffer and display to empty line
    */
    lb->current = 0;
    lb->lastchar = sizeof lb->line - 1;

    lb->lcurs = lb->line;
    lb->rcurs = lb->lineend = lb->line + lb->lastchar;
    *lb->lcurs = *lb->rcurs = '\0';

    lb->state = NORMAL;

    /*
	Reset history pointers to bottom
    */
    lb->lastline = 0;
    lb->matchlen = 0;

    /*
	Initialize flags
    */
    lb->flags &=~ (LD_DONE|LD_EOF|LD_INTR|LD_QUIT|LD_INSERT);
    lb->flags |= (LD_DIRTY);
    if (tbp->flags&TB_INSERT)
	lb->flags |= LD_INSERT;
    tbp->flags |= TB_READING;

#if INKERNEL
    tbp->iflag = tp->t_iflag;
    tbp->oflag = tp->t_oflag;
    tbp->lflag = tp->t_lflag;
    tbp->cflag = tp->t_cflag;
    for (c = 0; c < NCC + 2; c++)
	tbp->cc[c] = tp->t_cc[c];

    /*
	Print any pending output
    */
    while (tp->t_outq.c_cc != 0 || (tbp->flags&TB_WRITING))
    {
	kick_out(tp);
	delay(HZ/10);
    }
#endif

    while (!(lb->flags&LD_DONE))
    {
#if INKERNEL
	int                     ospl;

	ospl = spl6();
	{
	    /*
		Wait for input if there is none queued
	    */

	    if (tp->t_rawq.c_cc == 0)
	    {
		if (tbp->dorefresh)
		{
		    tbp->dorefresh = NO;	
		    splx(ospl);

		    reprint(lb);
		    kick_out(tp);
		    continue;
		}

		if (!(tbp->flags&TB_OPEN))
		{
		    lb->flags |= (LD_DONE|LD_EOF);
		    msg("Trying to cleread while CLED turned off",lb);
		    break;
		}

		if (tp->t_line != our_lineno || CLEDOFF(tp->t_lflag))
		{
		    tp->t_line = our_lineno;
		    tp->t_iflag = tbp->iflag;
		    tp->t_oflag = tbp->oflag;
		    tp->t_lflag = tbp->lflag;
		    tp->t_cflag = tbp->cflag;
		    for (c = 0; c < NCC + 2; c++)
			tp->t_cc[c] = tbp->cc[c];

		    tp->t_state |= ISOPEN;
		    tbp->dorefresh = YES;

		    msg("CLED: tty struct has been zapped. Resetting it.",lb);
		    continue;
		}

		/*
		    Sleep for a while for input to arrive
		*/
		time_id = timeout(read_timeout,tbp,HZ*15);
		tbp->readsleep = YES;

		if (sleep(tbp,(PZERO+8)|PCATCH))
		{
		    ospl = spl6();
		    {
			tbp->flags &=~ (TB_FLUSHIT|TB_READING);
			tbp->dorefresh = tbp->readsleep = NO;
		    }
		    splx(ospl);

		    untimeout(time_id);
		    u.u_error = EINTR;

		    return -1;
		}

		/*
		    Alleluiah! We may have gotten something from cleinput
		*/

		untimeout(time_id);
		splx(ospl);

		while (tp->t_outq.c_cc != 0 || (tbp->flags&TB_WRITING))
		{
		    kick_out(tp);
		    delay(HZ/10);
		}

		if (tbp->flags&TB_FLUSHIT)
		{
		    bol(lb);
		    d_eol(lb);

		    ospl = spl6();
		    {
			tbp->flags &=~ TB_FLUSHIT;
			tbp->dorefresh = tbp->readsleep;
		    }
		    splx(ospl);

		    if (!(tbp->flags & TB_OPEN))
			break;
		}
		continue;
	    }
	}
	splx(ospl);
#endif /* INKERNEL */

	/*
	    Get next char from the raw input queue, filled by cleinput
	*/

	c = cle_getc(tp);

	if (c != 0 && lb->state == NORMAL)
	{
	    if (c == tp->t_cc[VERASE])
		c = 0, lb->state = (NCC_SPC(VERASE));
	    else if (c == tp->t_cc[VKILL])
		c = 0, lb->state = (NCC_SPC(VKILL));
	    else if (c == tp->t_cc[VEOL] || c == tp->t_cc[VEOL2])
		c = 0, lb->state = (NCC_SPC(VEOL));
	    else if (c == tp->t_cc[VEOF])
		c = 0, lb->state = (NCC_SPC(VEOL)), lb->flags = LD_EOF;
	}

	lb->c = c;
	parse_it(lb);

#if INKERNEL
	kick_out(tp);
#endif
    }

    tbp->flags &=~ TB_READING;

    cp = lb->lcurs;
    if (!(lb->flags&(LD_EOF/*|LD_INTR|LD_QUIT*/)))
    {
	*cp++ = '\n';
	lb->lcurs = cp;
    }

    len = cp - lb->line;

#if INKERNEL
    if (len != 0 && !(lb->flags&(LD_INTR|LD_QUIT)))
    {
	if (len > u.u_count)
	    len = u.u_count;
	if (copyout(lb->line,u.u_base,len) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	u.u_base += len;
	u.u_count -= len;
	tp->t_col = 0;
	cle_putc('\r',tp);    /* ICRNL */
	cle_putc('\n',tp);    /* OCRNL */
    }

    /* count down the delimiter */
    if (tp->t_delct > 0)
	--tp->t_delct;

    if (tbp->broadcast.c_cc != 0)
    {
	send_brdcst(tbp,0);
	cle_putc('\r',tp);
	cle_putc('\n',tp);
    }

    kick_out(tp);
#else
    strncpy(u.u_base,lb->line,len);
    *(u.u_base + len) = 0;
#endif /* INKERNEL */

    lb->owed = (lb->line + len >= lb->lcurs) ? (unchar *) 0 : lb->line+len;

    lb->promptlen = 0;
    lb->prompt[0] = '\0';

    return 1;
}

#if INKERNEL
/*
    Write some text to the terminal but catch any trailing data into a prompt
    buffer. This routine is executed when no read is in progress on the
    terminal, an lb buffer is assigned to the terminal and a write comes
    through.

    ENTRY: tbp - ptr to tty_buf assigned to terminal lb - ptr to
    led_buf assigned to terminal u.u_base - ptr to message
    (u.u_count has length of message)

    EXIT: the text from the trailing \n (if any) of the message has been
    copied to the prompt buffer in the led_buf. If the last char of the
    message is a \n, then prompt buffer is left empty. If there are no
    \n's in the message, then the whole message is appended to any
    existing data in the prompt buffer. Chars are removed from the
    beginning of the buffer if the length of the new message exceeds the
    MAXPROMPT parameter.
*/

static void              catch_prompt(tbp,lb)
    struct tty_buf          *tbp;
    struct led_buf          *lb;
{
#if M_SPTALLOC
    unchar		    *prompt;
#else
    unchar		    prompt[MAXPROMPT+1];
#endif
    int			    maxlen;
    register unchar	    *tail,*old;
    register int	    headlen;

    if (tbp->broadcast.c_cc != 0)
	send_brdcst(tbp,1);

    /*
	If the write is zero length, we just return; if the last
	character of the write is a newline, than the write is surely
	not for a prompt; prompts are non NL terminated writes.
    */

    {
	unchar           ch;

	if (copyin(u.u_base + u.u_count - 1,&ch,sizeof ch) < 0)
	    return;

	if (ch == '\n')
	{
	    lb->promptlen = 0;
	    lb->prompt[0] = '\0';
	    return;
	}
    }

#if M_SPTALLOC
    prompt = (unchar *) Sptalloc(maxlen+1);
#endif

    /*
	Fetch the last maxlen characters of the prompt, and null
	terminate them.
    */
    maxlen = MIN(u.u_count,MAXPROMPT); /* This is at least 1 */
    if (copyin(u.u_base + (u.u_count - maxlen),prompt,maxlen) < 0)
	return;
    *(tail = prompt + maxlen) = '\0';

    /*
	Determine the real prompt, which is the tail after the last
	'\r' or '\n'. We *know* that maxlen != 0 implies tail > prompt initially.
    */
    do --tail; while (tail > prompt && !(*tail == '\r' || *tail == '\n'));
    if (*tail == '\r' || *tail == '\n') tail++;

    /*
        First we assume that the head of the new prompt is empty, then we
        check; if the old prompt was not empty and the new prompt tail did not
        start with '\r' or '\n', and leaves some space in the prompt buffer we
        prepend the tail of the old prompt as the head of the new one,
        shifting it into place if necessary. We expect this to be executed
        very rarely; actually we should probably not bother at all...
    */

    old = lb->prompt;

    if (tail == prompt && lb->promptlen && !(tbp->flags&TB_WRITING)
	&& (headlen = tail-prompt + MAXPROMPT-maxlen) > 0)
    {
	if (headlen >= lb->promptlen)
	    old += lb->promptlen;
	else
	{
	    bcopy(old+lb->promptlen-headlen,old,headlen);
	    old += headlen;
	}
    }

    /*
	A simple while will do instead of bcopy; we assume the tail will
	usually be very small, i.e. <= 8 chars.
    */
    while (*old++ = *tail++);
    lb->promptlen = old-1 - lb->prompt;

#if M_SPTALLOC
    Sptfree(temp,maxlen+1);
#endif
}

/*
    Breakthru. This routine is called when a message is to be sent to the
    terminal while a read is in progress. The message is prefixed with a
    \r<clear_to_eol> to make room on the current line for the new message and
    all text up to and including the last \n is transmitted to the terminal.
    Any text from the last \n to the end of the buffer is saved in the
    broadcast clist which is transmitted either with the next message to be
    written or when the read completes. In any case, the refresh bit is set to
    cause the user's command line to be reprinted after the write completes.

    ENTRY: tbp - ptr to tty_buf u.u_base - ptr to message to send to
    terminal.

    EXIT: user data is transmitted to the terminal.
*/

#if BREAKTHRU
static void             breakthru(tbp)
    struct tty_buf          *tbp;
{
    unchar                  last;
    int                     len;
    struct clist            *bcl;

    if (copyin(u.u_base + u.u_count - 1,&last,1) < 0)
	return;

    bcl = &tbp->broadcast;

    if (last == '\n')
    {
	if (bcl->c_cc > 0)
	    send_brdcst(tbp,YES);
	else
	{
	    cle_putc('\r',tbp->ttyp);
	    cle_puts(tbp->tcap[TCAP_CLREOL],tbp->ttyp);
	}
	tbp->dorefresh = YES;
    }
    else
    {
	int                     len,oldlen;
	unchar                  *src,*dst;
#if M_SPTALLOC
	unchar                  *temp;
#else
	unchar                  temp[MAXPROMPT];
#endif

	len = 132;		/* assume max length */
	if (len <= u.u_count)
	{
	    if (bcl->c_cc > 0)
		send_brdcst(tbp,1);
	}
	else
	{
	    /* user data is smaller than 1 line */

	    /* but would overflow */
	    if (bcl->c_cc + u.u_count > len)
	    {
		send_brdcst(tbp,1);
		cle_putc('\r',tbp->ttyp);
		cle_putc('\n',tbp->ttyp);
		tbp->dorefresh = YES;
	    }

	    len = u.u_count;
	}

	oldlen = len;

#if M_SPTALLOC
	temp = (unchar *) Sptalloc(oldlen + 1);
#endif

	if (copyin(u.u_base + (u.u_count - len),temp,len) < 0)
	    return;

	src = temp + len;
	*src = '\0';

	for (; len > 0; --len)
	{
	    unchar                  c;

	    c = *--src;
	    if (c == '\r' || c == '\n')
	    {
		++src;
		break;
	    }
	}

	/* compute real length of message */
	len = oldlen - len;
	if (bcl->c_cl != 0)
	{
	    int                     i;
	    struct cblock           *cbkp;

	    cbkp = bcl->c_cl;
	    i = CLSIZE - cbkp->c_last;	/* room in last cblock */
	    if (i > 0)
	    {
		dst = &cbkp->c_data[cbkp->c_last];
		if (i > len)
		    i = len;
		len -= i;
		u.u_count -= i;
		bcl->c_cc += i;
		/* move end ptr */
		cbkp->c_last += i;
		while (i-- > 0)
		    *dst++ = *src++;
	    }
	}

	if (len > 0)
	{
	    int                     ospl;
	    struct clist            tcl;

	    /* init our dummy clist */
	    tcl.c_cc = 0;
	    tcl.c_cf = tcl.c_cl = 0;
	    u.u_count -= len;

	    ospl = spl6();
	    {
		putcbp(&tcl,src,len);
	    }
	    splx(ospl);

	    /* if broadcast is empty */
	    if (bcl->c_cf == 0)
		bcl->c_cf = tcl.c_cf;
	    else
		bcl->c_cl->c_next = tcl.c_cf;

	    bcl->c_cc += tcl.c_cc;
	    bcl->c_cl = tcl.c_cl;
	}

#if M_SPTALLOC
	Sptfree(temp,oldlen + 1);
#endif
    }

    kick_out(tbp->ttyp);
}
#endif /* BREAKTHRU */

/*
    Write some text to the terminal. Writes to the terminal can occur at any
    time by any suitably privledged process. An attempt is made to determine
    what writes constitute the output of a "prompt" string. This is done by
    capturing and remembering in the brodcast clist a copy of the data to
    write from the last \r or \n in the message to the end of the message.
    Data that has no \r or \n in it is appended to any existing data in the
    clist (often processes do single char output to the terminal so its no
    wonder the system gets slow at times). If a led_buf has been assigned to
    the process doing the write, then the "prompt" data is placed in the
    led_buf instead of the broadcast clist. If a read is pending on the
    terminal when a write is issued, only the data up to and including the
    last \n is transmitted. The remainder is saved in the broadcast clist. If
    data ends up being sent to the terminal, then the refresh bit is set and
    the read process is awakened (which causes broadcast messages to
    automatically refresh the input line).

    ENTRY: Process context: task. td - ptr to tty struct

    EXIT:  Write data sent to the terminal and/or stored in the broadcast
    clist or led_buf (if one assigned).
*/

clewrite(td)
    struct tty              *td;
{
    int			    ospl;
    struct led_buf	    *lb;
    struct tty_buf	    *tbp;

    if (CLEDOFF(td->t_lflag))
    {
	(*linesw[0].l_write)(td);
	return;
    }

    tbp = find_ttybuf(td);
    if (tbp != 0)
    {
#if BREAKTHRU
	if (tbp->flags&TB_READING)
	    breakthru(tbp);
	else
#else
	if (!(tbp->flags&TB_READING))
#endif
	{
#if !MULTILB
	    lb = tbp->lbp;
	    if (u.u_count > 0)
		catch_prompt(tbp,lb);
#else
	    lb = find_ledbuf(tbp,u.u_procp);
	    if (lb != 0)
	    {
		if (u.u_count > 0)
		    catch_prompt(tbp,lb);
	    }
	    else
	    {
		if (tbp->broadcast.c_cc > 0)
		    send_brdcst(tbp,0);
	    }
#endif
	}
	tbp->flags |= TB_WRITING;
    }

    if (u.u_count > 0)
	    (*linesw[0].l_write)(td);
    else    kick_out(td);

    if (tbp != 0)
    {
	tbp->flags &=~ TB_WRITING;

	ospl = spl6();
	{
	    if (tbp->dorefresh && tbp->readsleep)
	    {
		tbp->readsleep = NO;
		wakeup(tbp);
	    }
	}
	splx(ospl);
    }
}

#if VERBOSE
static void             show_pid(disp,str,td)
    int                     disp;
    unchar                  *str;
    struct tty              *td;
{
    unchar                  tmp[10],*s;

    if (disp != 0)
    {
	cle_puts("\rcled version ",td);
	cle_puts(VERSION,td);
	cle_putc('.',td);
	s = itoa(PATCHLEVEL,tmp,10); *s = '\0'; cle_puts(s,td);
	cle_putc(' ',td);
    }
    cle_puts(str,td);
    s = itoa(u.u_procp->p_pid,tmp,10); *s = '\0'; cle_puts(tmp,td);
    cle_puts(" (",td);
    s = itoa(u.u_procp->p_ppid,tmp,10); *s = '\0'; cle_puts(tmp,td);
    cle_puts(")\r\n",td);
    return;
}
#endif /* VERBOSE */

/*
    Line discipline IOCTL routine.

    ENTRY: Process context: task. td - ptr to tty struct f1 - function to
    perform

    EXIT: ioctl function is performed
*/

cleioctl(td,f1,arg,mode)
    struct tty              *td;
    int                     f1,mode;
    faddr_t                 arg;
{
    struct tty_buf          *tbp;
    struct led_buf          *lb;

    tbp = find_ttybuf(td);
    if (tbp != 0
	&& ((tbp->flags&(TB_READING|TB_WRITING)) || !tbp->readsleep))
    {
#if VERBOSE
	cle_puts("CLED: ioctl issued while terminal busy. stty bits may be zapped",tbp->tbp);
#endif
	kick_out(td);
    }

    if (f1 < LDGETCOLS)
    {
	if (f1 == LDCLOSE)
	    close_it(tbp,td);
	else if (f1 == LDOPEN || f1 == LDCHG)
	    open_it(tbp,td);
	(*linesw[0].l_ioctl) (td,f1,arg,mode);
	kick_out(td);
	return;
    }

#if CLEDIO
    
    garbage_collect(tbp);

    if (tbp == 0)
	tbp = get_ttybuf(td);

    if (tbp == 0)
    {
	u.u_error = ERR_NOTTYBUF;
	return;
    }

#if MULTILB
    if ((lb = tbp->lbp) != 0)
    {
	struct led_buf          *us = 0,*parent = 0;

	do
	{
	    if (u.u_procp->p_ppid == lb->pid)   parent = lb;
	    if (u.u_procp->p_pid == lb->pid)    us = lb;
	}
	while ((lb = lb->next) != tbp->lbp);

	lb = (us != 0) ? us : (parent != 0) ? parent : 0;
    }
#else
    lb = tbp->lbp;
#endif

    switch (f1)
    {
    case LDGETCOLS:
	if (copyout(&tbp->cols,arg,sizeof tbp->cols) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	break;

    case LDSETCOLS:
    {
	int			    cols;

	if (copyin(arg,&cols,sizeof cols) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	setcols(tbp,cols);
    }
	break;

    case LDGETBF:
    {
	struct set_key          sk;
	char                    cnt,len;
	faddr_t                 outp;

	sk.modes = 0;
	sk.modes |= (tbp->flags&TB_INSERT) ? CLEMODE_INSERT : CLEMODE_OVER;
	sk.kdbuf_len = CLEKEY_MAX;

	if (copyout(tbp->keymap,arg + sizeof (sk),CLEKEY_MAX) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	outp = arg + sizeof (sk) + CLEKEY_MAX;

	for (cnt = 0; cnt < TCAP_COUNT; ++cnt)
	{
	    unchar                  *s;

	    if (copyout(&cnt,outp++,1) < 0)
	    {
		u.u_error = EFAULT;
		return;
	    }
	    s = (unchar *) tbp->tcap[cnt];
	    while (*s++);

	    len = s - (unchar *) tbp->tcap[cnt];
	    if (copyout(tbp->tcap[cnt],outp,len) < 0)
	    {
		u.u_error = EFAULT;
		return;
	    }
	    outp += len;
	}

	sk.tcapbuf_len = outp - (arg + sizeof (sk) + CLEKEY_MAX);
	if (copyout(&sk,arg,sizeof (sk)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
    }
	break;

    case LDSETBF:
    {
	struct set_key          sk;
	int                     oldflag;

	if (copyin(arg,&sk,sizeof (sk)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	oldflag = tbp->flags;
	if (sk.modes & CLEMODE_INSERT)	tbp->flags |= TB_INSERT;
	else if (sk.modes & CLEMODE_OVER)	tbp->flags &= ~TB_INSERT;

	if (sk.kdbuf_len > CLEKEY_MAX)
	{
	    sk.kdbuf_len = CLEKEY_MAX;
	    if (copyout(&sk,arg,sizeof (sk)) < 0)
	    {
		u.u_error = EFAULT;
		return;
	    }
	    u.u_error = ERR_BADPARAM;
	    return;
	}

	if (sk.kdbuf_len > 0)
	{
#if M_SPTALLOC
	    unchar                  *tmp,*kp;
#else
	    unchar                  tmp[100],*kp;
#endif
	    int                     cnt,size;

	    size = sk.kdbuf_len * 2;
#if M_SPTALLOC
	    kp = tmp = (unchar *) Sptalloc(size);
#else
	    if (size > sizeof (tmp))
	    {
		u.u_error = ERR_BADPARAM;
		return;
	    }
	    kp = tmp;
#endif

	    if (copyin(arg + sizeof (sk),tmp,size) < 0)
	    {
		u.u_error = EFAULT;
		return;
	    }
	    for (cnt = 0; cnt < sk.kdbuf_len; ++cnt)
	    {
		int                     key,func;

		key = *kp++;
		func = *kp++;
		if (key >= CLEKEY_MAX || func >= CLEFUN_MAX)
		{
#if M_SPTALLOC
		    Sptfree(tmp,size);
#endif
		    sk.kdbuf_len = cnt;
		    copyout(&sk,arg,sizeof (sk));
		    u.u_error = ERR_BADPARAM;
		    return;
		}
		tbp->keymap[CLEKEY_CMD(key)] = func;
	    }
#if M_SPTALLOC
	    Sptfree(tmp,size);
#endif
	}

	if (sk.tcapbuf_len > 0)
	{
	    setup_tcap_defaults(tbp);
	    if (sk.tcapbuf_len > 1)
	    {
		unchar                  *s;
		int                     nfg = 0;
#if M_SPTALLOC
		s = tbp->tcstrings = (unchar *) Sptalloc(sk.tcapbuf_len);
#else
		s = tbp->tcstrings;
		if (sk.tcapbuf_len > TCAP_SIZE)
		{
		    sk.tcapbuf_len = TCAP_SIZE;
		    copyout(&sk,arg,sizeof (sk));
		    u.u_error = ERR_BADPARAM;
		    return;
		}
#endif
		tbp->tclen = sk.tcapbuf_len;
		if (copyin(arg + sizeof (sk) + sk.kdbuf_len*2,s,tbp->tclen) < 0)
		{
		    u.u_error = EFAULT;
		    return;
		}
		while (s < tbp->tcstrings + tbp->tclen)
		{
		    unchar                  *ap;
		    int                     str;

		    str = *s++;
		    if (str >= TCAP_COUNT)
		    {
			nfg = 1;
			break;
		    }
		    ap = s;
		    while (*ap && ap < tbp->tcstrings + tbp->tclen)
			++ap;
		    if (*ap != 0)
		    {
			nfg = 1;
			break;
		    }

		    tbp->tcap[str] = (char *) s;
		    s = ap + 1;
		}
		if (nfg)
		{
		    sk.tcapbuf_len = s - tbp->tcstrings;
		    copyout(&sk,arg,sizeof (sk));
		    u.u_error = ERR_BADPARAM;
		    return;
		}
	    }
	}
    }
	break;

    case LDGETHB:
	if (lb == 0)
	{
	    u.u_error = ERR_NOLBASS;
	    return;
	}

	if (copyout(lb->history,arg,MAXHISTORY) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	break;

    default:
	u.u_error = ERR_BADIOCTL;
    }
#endif /* CLEDIO */
}

/*
    Device driver entry points; used for backdoor setup of CLED.
*/

cledinit(dev)
    int                     dev;
{
    int                     cnt;

    for (cnt = 0; cnt < linecnt; ++cnt)
	if (linesw[cnt].l_open == cleopen)
	    break;

    if (cnt < linecnt)
    {
	our_lineno = cnt;
	printf("CLED %s PL%d is ldisc %d.\n",VERSION,PATCHLEVEL,cnt);
	garbage_collect((struct tty_buf *) 0);
    }
    else
    {
#if M_UNIX
	cmn_err(CE_WARN,"CLED %s not in linesw table. Not installed",VERSION);
#else
	printf("CLED %s not in linesw table. Not installed.\n",VERSION);
#endif
    }
    return;
}

cledioctl(dev,cmd,arg,mode)
    int                     dev,cmd,mode;
    faddr_t                 arg;
{
    switch (cmd)
    {

#if CLEDIO
    case LDGETS:
    {
	int                     cnt;
	unchar                  *s;
	struct led_buf          *lb;
	struct tty_buf          *tbp;
	struct cle_stats        sts;

	sts.line	= our_lineno;
	sts.ledbufs	= cle_leds;
	sts.ttybufs	= cle_ttys;
	sts.promptsize	= MAXPROMPT;
	sts.linesize	= MAXLINE;
	sts.histsize	= MAXHISTORY;
	sts.multi_lb	= MULTILB;
	sts.spt		= M_SPTALLOC;
#if M_SPTALLOC
	sts.tcapsize	= 256;
#else
	sts.tcapsize	= TCAP_SIZE;
#endif

	s = (unchar *) VERSION;
	for (cnt = 0; cnt < 4; ++cnt)
	{
	    if ((sts.vers[cnt] = *s++) == 0)
		break;
	}

	tbp = tty_free;
	cnt = 0;
	if (tbp != 0)
	{
	    do
	    {
		tbp = tbp->next;
		++cnt;
	    } while (tbp != tty_free && cnt < cle_ttys);
	}
	sts.ttybufs_used = cle_ttys - cnt;

#if !MULTILB
	sts.ledbufs_used = sts.ttybufs_used;
#else
	cnt = 0;
	lb = ldb_free;
	if (lb != 0)
	{
	    do
	    {
		lb = lb->next;
		++cnt;
	    }
	    while (lb != ldb_free && cnt < cle_leds);
	}

	sts.ledbufs_used = cle_leds - cnt;
#endif

	if (copyout(&sts,arg,sizeof (struct cle_stats)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
    }
	break;
#endif /* CLEDIO */

#if DEBUG
    case LDGETB:
    {
	struct cle_buf          bfs;

	bfs.lbsize = cle_leds * sizeof (struct led_buf);
	bfs.tbsize = cle_ttys * sizeof (struct tty_buf);

	bfs.lbbase = cle_buffers;
#if MULTILB
	bfs.lbfree = ldb_free;
#else
	bfs.lbfree = 0;
#endif

	bfs.tbbase = cle_ttybuf;
	bfs.tbused = tty_used;
	bfs.tbfree = tty_free;

	bfs.procbase = proc;

	if (copyout(&bfs,arg,sizeof (struct cle_buf)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	arg += sizeof (struct cle_buf);

	if (copyout(cle_buffers,arg,bfs.lbsize) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	arg += bfs.lbsize;

	if (copyout(cle_ttybuf,arg,bfs.tbsize) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
    }
	break;

    case LDGETTTY:
    {
	struct tty              *ttyp;

	if (copyin(arg,&ttyp,sizeof (ttyp)) < 0
	    || copyout(ttyp,arg,sizeof (struct tty)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
    }
	break;

    case LDGETC:
    {
	struct clist            *clp;
	struct cblock           *cbp;
	int                     cnt;

	if (copyin(arg,&clp,sizeof (clp)) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
	if (clp->c_cc != 0)
	{
	    int                     ospl;

	    ospl = spl6();
	    {
		cbp = clp->c_cf;
		do
		{
		    unchar                  siz;

		    siz = cbp->c_last - cbp->c_first;
		    if (siz != 0)
		    {
			if (copyout(&siz,arg,1) < 0)
			{
			    u.u_error = EFAULT;
			    return;
			}
			++arg;
			if (copyout(&cbp->c_data[cbp->c_first],arg,siz) < 0)
			{
			    u.u_error = EFAULT;
			    return;
			}
			arg += siz;
		    }
		    cbp = cbp->c_next;
		} while (cbp != 0);
	    }
	    splx(ospl);
	}

	cnt = 0;
	if (copyout(&cnt,arg,1) < 0)
	{
	    u.u_error = EFAULT;
	    return;
	}
    }
	break;
#endif /* M_DEBUG */
    default:
	u.u_error = ERR_BADIOCTL;
    }
}

#else /* INKERNEL */

static unchar           cmd_line[256];
static struct tty       dum_tty;

struct termio           ostate;	/* saved tty state */
struct termio           nstate;	/* values for editor mode */

main()
{
    int                     c;

    dum_tty.t_cc[VINTR]	    = 'C' & 0x1f;
    dum_tty.t_cc[VQUIT]	    = '\\' & 0x1f;
    dum_tty.t_cc[VERASE]    = RUB;
    dum_tty.t_cc[VKILL]	    = 'U' & 0x1f;
    dum_tty.t_cc[VEOF]	    = 'D' & 0x1f;
    dum_tty.t_cc[VEOL]	    = 'M' & 0x1f;

    ioctl(0,TCGETA,&ostate);
    ioctl(0,TCGETA,&nstate);

    nstate.c_iflag	= 0;
    nstate.c_oflag	= 0;
    nstate.c_lflag	= 0;
    nstate.c_cc[VEOF]	= 1;
    nstate.c_cc[VEOL]	= 0;

    ioctl(0,TCSETA,&nstate);

    u.u_base	= cmd_line;
    u.u_count	= sizeof (cmd_line);

    garbage_collect((struct tty_buf *) 0);

    tty_used	= cle_ttybuf;
    get_ttybuf(&dum_tty);

    cle_puts("Outputting controls\r\n",&dum_tty);

    while (cleread(&dum_tty) && strlen(cmd_line))
	fprintf(stderr,"\r\nRead %d chars\r\n",strlen(cmd_line));

    ioctl(0,TCSETA,&ostate);
    return;
}

#endif /* INKERNEL */
